//-----------------------------------------------------------------------------
// Demonstrates use of the Fpu library for interfacing with the uM-FPU V3.1
// floating point coprocessor. This example takes an integer value representing
// the diameter of a circle in centimeters, converts the value to inches,
// then calculates circumference and area. The results are displayed on the
// serial monitor.
//
// @author Cam Thompson, Micromega Corporation
// <www.micromegacorp.com>
//
// @version
//  October 2, 2011
//  - updated for new SPI and FpuSerial libraries
//  December 15, 2008
//  - original version
//-----------------------------------------------------------------------------

#include <SPI.h>
#include <Fpu.h>
#include <FpuSerial.h>

//-------------------- uM-FPU register definitions ----------------------------

#define DiameterIn      10				// diameter in inches
#define Circumference	11				// circumference
#define Area            12				// area
#define Pi              13				// constant pi

void setup()
{
  Serial.begin(9600);
  Serial.println("Sample");
  
  SPI.begin();
  Fpu.begin();
  
  if (Fpu.sync() == SYNC_CHAR)
    FpuSerial.printVersionln();
  else
  {
    Serial.println("FPU not detected");
    while(1) ; // stop if FPU not detected
  }  
}

void loop()
{
  byte diameterCm;

  // Load constant for later use.
  Fpu.write(SELECTA, Pi, LOADPI, FSET0);

  // Get diameter in centimeters.  The value would typically come from a sensor
  // reading, but in this example an assumed value of 25 is used.
  diameterCm = 25;
  Serial.print("\r\nDiameter (cm):       ");
  Serial.println(diameterCm, DEC);

  // Convert inches to centimeters
  Fpu.write(SELECTA, DiameterIn, FSETI, diameterCm);
  Fpu.write(FCNV, 5);
  Serial.print("Diameter (in.):      ");
  FpuSerial.printFloatln(0);

  // circumference = diameter * pi
  Fpu.write(SELECTA, Circumference, FSET, DiameterIn);
  Fpu.write(FMUL, Pi);
  Serial.print("Circumference (in.): ");
  FpuSerial.printFloatln(0);

  // area = (diameter / 2)^2 * pi
  Fpu.write(SELECTA, Area, FSET, DiameterIn);
  Fpu.write(FDIVI, 2);
  Fpu.write(FMUL, Area, FMUL, Pi);
  Serial.print("Area (sq.in.):       ");
  FpuSerial.printFloatln(0);

  Serial.println("\r\nDone.");
  while(1) ;
}
